<?php
/**
 * Plugin Name: ContentBeast Auto Publisher (Lite)
 * Description: Receives ContentBeast article JSON and creates WordPress posts with tags, category, and featured image. Includes daily cron job for scheduled publishing.
 * Version: 1.1.0
 * Author: ContentBeast
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

final class ContentBeast_Auto_Publisher_Lite {
    const OPT_GROUP = 'contentbeast_auto_pub_opts_group';
    const OPT_NAME  = 'contentbeast_auto_pub_opts';
    const NS        = 'contentbeast/v1';
    const ROUTE     = '/article';
    const CRON_HOOK = 'contentbeast_daily_publish_cron';

    public function __construct() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'register']);
        add_action('rest_api_init', [$this, 'rest']);
        add_action('init', [$this, 'setup_cron']);
        add_action(self::CRON_HOOK, [$this, 'daily_cron_job']);
        
        // Clean up cron on deactivation
        register_deactivation_hook(__FILE__, [$this, 'deactivate_cron']);
    }

    private function opts() {
        $defaults = [
            'contentbeast_api_key'   => '',
            'webhook_url'            => '',
            'default_status'         => 'publish', // Always publish as per requirements
            'default_author'         => 1,       // WP user ID
            'category_fallback'      => '',
            'trust_incoming_status'  => '1',
            'cron_enabled'           => '1',
            'cron_time'              => '02:00',  // 2am CST
            'last_cron_run'          => ''
        ];
        return wp_parse_args(get_option(self::OPT_NAME, []), $defaults);
    }

    public function menu() {
        add_options_page(
            'ContentBeast Publisher',
            'ContentBeast Publisher',
            'manage_options',
            'contentbeast-auto-publisher-lite',
            [$this, 'settings_page']
        );
    }

    public function register() {
        register_setting(self::OPT_GROUP, self::OPT_NAME);

        add_settings_section('m_main', 'API Configuration', function () {
            echo '<p>Configure your connection with ContentBeast.com below.</p>';
        }, 'contentbeast-auto-publisher-lite');

        add_settings_section('m_webhook', 'Webhook Configuration', function () {
            $webhook_url = rest_url(self::NS . self::ROUTE);
            echo '<p><strong>Your Webhook URL:</strong></p>';
            echo '<p><code style="background:#f1f1f1;padding:10px;display:block;word-break:break-all;">' . esc_html($webhook_url) . '</code></p>';
            echo '<p>Copy this URL and add it to your ContentBeast.com integration settings.</p>';
            echo '<p>Send JSON via POST and include header <code>X-Content-Beast-Signature: sha256=&lt;hmac-hex&gt;</code> where HMAC is computed on the raw request body using your API Key.</p>';
        }, 'contentbeast-auto-publisher-lite');

        add_settings_section('m_cron', 'Scheduled Publishing', function () {
            $o = $this->opts();
            echo '<p>Daily cron job runs at 2am CST to publish scheduled articles.</p>';
            if (!empty($o['last_cron_run'])) {
                echo '<p><strong>Last cron run:</strong> ' . esc_html($o['last_cron_run']) . '</p>';
            }
        }, 'contentbeast-auto-publisher-lite');

        add_settings_field('contentbeast_api_key', 'ContentBeast.com API Key', function () {
            $o = $this->opts();
            echo '<input type="password" style="width:420px" name="'.esc_attr(self::OPT_NAME).'[contentbeast_api_key]" value="'.esc_attr($o['contentbeast_api_key']).'" required>';
            echo '<p class="description">Get this from your ContentBeast.com WP integration page</p>';
        }, 'contentbeast-auto-publisher-lite', 'm_main');

        add_settings_field('default_author', 'Default Author (user ID)', function () {
            $o = $this->opts();
            echo '<input type="number" min="1" style="width:120px" name="'.esc_attr(self::OPT_NAME).'[default_author]" value="'.esc_attr($o['default_author']).'">';
        }, 'contentbeast-auto-publisher-lite', 'm_main');

        add_settings_field('category_fallback', 'Fallback Category (name)', function () {
            $o = $this->opts();
            echo '<input type="text" style="width:280px" name="'.esc_attr(self::OPT_NAME).'[category_fallback]" value="'.esc_attr($o['category_fallback']).'">';
            echo '<p class="description">Used when no category is provided in the article data</p>';
        }, 'contentbeast-auto-publisher-lite', 'm_main');

        add_settings_field('cron_enabled', 'Enable Daily Cron Job', function () {
            $o = $this->opts(); ?>
            <label>
                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NAME); ?>[cron_enabled]" value="1" <?php checked($o['cron_enabled'],'1'); ?>>
                Run daily cron job at 2am CST to publish scheduled articles
            </label>
        <?php }, 'contentbeast-auto-publisher-lite', 'm_cron');

        add_settings_field('trust_incoming_status', 'Trust incoming status/publish_at', function () {
            $o = $this->opts(); ?>
            <label>
                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NAME); ?>[trust_incoming_status]" value="1" <?php checked($o['trust_incoming_status'],'1'); ?>>
                Allow payload fields <code>status</code> and <code>publish_at_iso8601</code> to control draft/publish/schedule.
            </label>
        <?php }, 'contentbeast-auto-publisher-lite', 'm_main');
    }

    public function settings_page() {
        $o = $this->opts(); 
        $webhook_url = rest_url(self::NS . self::ROUTE);
        $logo_url = plugins_url('contentbeast-logo.png', __FILE__);
        ?>
        <style>
            .contentbeast-header {
                background: #fff;
                border: 1px solid #ddd;
                padding: 20px 25px;
                margin: 20px 0;
                border-radius: 6px;
                box-shadow: 0 1px 3px rgba(0,0,0,0.05);
            }
            .contentbeast-header-content {
                display: flex;
                align-items: center;
                gap: 20px;
            }
            .contentbeast-logo {
                width: 60px;
                height: 60px;
                flex-shrink: 0;
            }
            .contentbeast-header-text h2 {
                margin: 0 0 8px 0;
                font-size: 22px;
                font-weight: 600;
                color: #1d2327;
            }
            .contentbeast-header-text p {
                margin: 0 0 10px 0;
                color: #50575e;
                font-size: 14px;
                line-height: 1.5;
            }
            .contentbeast-header-link {
                display: inline-block;
                color: #2271b1;
                text-decoration: none;
                font-weight: 500;
                font-size: 14px;
            }
            .contentbeast-header-link:hover {
                color: #135e96;
                text-decoration: underline;
            }
            .contentbeast-save-button {
                background: #2271b1 !important;
                border-color: #2271b1 !important;
                color: #fff !important;
                text-decoration: none;
                text-shadow: none;
                font-size: 16px !important;
                line-height: 1.5 !important;
                padding: 12px 24px !important;
                height: auto !important;
                border-radius: 4px;
                cursor: pointer;
                border-width: 1px;
                border-style: solid;
                font-weight: 600;
                display: inline-block;
                margin: 15px 0 !important;
            }
            .contentbeast-save-button:hover,
            .contentbeast-save-button:focus {
                background: #135e96 !important;
                border-color: #135e96 !important;
                color: #fff !important;
            }
            .contentbeast-save-button:active {
                transform: translateY(1px);
            }
        </style>
        <div class="wrap">
            <h1>ContentBeast Auto Publisher (Lite)</h1>
            
            <!-- ContentBeast Header with Logo -->
            <div class="contentbeast-header">
                <div class="contentbeast-header-content">
                    <img src="<?php echo esc_url($logo_url); ?>" alt="ContentBeast Logo" class="contentbeast-logo">
                    <div class="contentbeast-header-text">
                        <h2>Automate blog writing with AI. Save 2-3 hours a day.</h2>
                        <p>Get mentioned by ChatGPT and rank on Google. Automate high-quality, SEO and AI-optimized blog articles to your website.</p>
                        <a href="https://contentbeast.com" target="_blank" rel="noopener noreferrer" class="contentbeast-header-link">Visit ContentBeast.com →</a>
                    </div>
                </div>
            </div>
            
            <form method="post" action="options.php">
                <?php settings_fields(self::OPT_GROUP); ?>
                
                <!-- Top Save Button -->
                <input type="submit" name="submit" id="submit-top" class="contentbeast-save-button" value="Save Changes">
                
                <?php do_settings_sections('contentbeast-auto-publisher-lite'); ?>
                
                <!-- Bottom Save Button -->
                <input type="submit" name="submit" id="submit-bottom" class="contentbeast-save-button" value="Save Changes">
            </form>

            <div style="background: #f9f9f9; padding: 15px; margin: 20px 0; border-left: 4px solid #0073aa;">
                <h3>Setup Instructions</h3>
                <ol>
                    <li><strong>Get API Key:</strong> Generate an API key from your ContentBeast.com WP integration page</li>
                    <li><strong>Add API Key:</strong> Enter the API key in the field above and save settings</li>
                    <li><strong>Configure Webhook:</strong> Copy the webhook URL below and add it to your ContentBeast.com settings</li>
                    <li><strong>Enable Cron:</strong> Check the cron job option above to enable daily scheduled publishing at 2am CST</li>
                </ol>
            </div>

            <h2>Test with curl</h2>
<pre>curl -X POST "<?php echo esc_html($webhook_url); ?>" \
  -H "Content-Type: application/json" \
  -H "X-Content-Beast-Signature: sha256=&lt;hmac-hex&gt;" \
  -d '{
    "event":"article.created",
    "article":{
      "title":"AI Content Strategy for SaaS",
      "slug":"ai-content-strategy-saas",
      "html":"&lt;h1&gt;AI Content Strategy&lt;/h1&gt;&lt;p&gt;...&lt;/p&gt;",
      "excerpt":"Quick guide...",
      "tags":["ai","saas","seo"],
      "category":"Guides",
      "featured_image_url":"https://example.com/img.jpg",
      "status":"publish",
      "publish_at_iso8601": null,
      "canonical_url":"https://yoursite.com/blog/ai-content-strategy-saas"
    }
  }'</pre>
            <p>Compute HMAC with <code>hash_hmac('sha256', $raw_json, $api_key)</code> and send as hex.</p>

            <h2>Cron Job Status</h2>
            <p><strong>Next scheduled run:</strong> <?php echo esc_html($this->get_next_cron_time()); ?></p>
            <?php if ($o['cron_enabled'] === '1'): ?>
                <p style="color: green;">✓ Cron job is enabled</p>
            <?php else: ?>
                <p style="color: orange;">⚠ Cron job is disabled</p>
            <?php endif; ?>
        </div>
    <?php }

    public function setup_cron() {
        $o = $this->opts();
        
        if ($o['cron_enabled'] === '1') {
            if (!wp_next_scheduled(self::CRON_HOOK)) {
                // Schedule for 2am CST daily
                $timestamp = strtotime('tomorrow 2:00 AM');
                wp_schedule_event($timestamp, 'daily', self::CRON_HOOK);
            }
        } else {
            // Remove cron if disabled
            $timestamp = wp_next_scheduled(self::CRON_HOOK);
            if ($timestamp) {
                wp_unschedule_event($timestamp, self::CRON_HOOK);
            }
        }
    }

    public function deactivate_cron() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
    }

    public function daily_cron_job() {
        $o = $this->opts();
        
        // Update last run time
        $opts = $o;
        $opts['last_cron_run'] = current_time('mysql');
        update_option(self::OPT_NAME, $opts);
        
        // Log cron run
        error_log('ContentBeast Auto Publisher: Daily cron job executed at ' . current_time('mysql'));
        
        // Here you would typically make an API call to ContentBeast to get scheduled articles
        // For now, we'll just log that the cron ran
        // In a future version, you could add API integration to fetch and publish scheduled content
        
        do_action('contentbeast_daily_cron_executed');
    }

    private function get_next_cron_time() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            return date('Y-m-d H:i:s', $timestamp);
        }
        return 'Not scheduled';
    }

    public function rest() {
        register_rest_route(self::NS, self::ROUTE, [
            'methods'  => 'POST',
            'callback' => [$this, 'handle'],
            'permission_callback' => '__return_true'
        ]);
    }

    private function verify_sig($raw, $secret, $header) {
        if (!$secret || !$header) return false;
        $parts = explode('=', $header, 2);
        if (count($parts) !== 2 || strtolower($parts[0]) !== 'sha256') return false;
        $expected = hash_hmac('sha256', $raw, $secret);
        return hash_equals($expected, $parts[1]);
    }

    public function handle(WP_REST_Request $req) {
        $o   = $this->opts();
        $raw = $req->get_body();
        $sig = $req->get_header('x-content-beast-signature');

        if (!$this->verify_sig($raw, $o['contentbeast_api_key'], $sig)) {
            return new WP_REST_Response(['error'=>'invalid_signature'], 401);
        }

        $data = json_decode($raw, true);
        if (!is_array($data) || empty($data['article'])) {
            return new WP_REST_Response(['error'=>'invalid_payload'], 400);
        }

        $a = $data['article'];
        $title   = isset($a['title']) ? wp_strip_all_tags($a['title']) : '(untitled)';
        $slug    = isset($a['slug']) ? sanitize_title($a['slug']) : '';
        $content = isset($a['html']) ? wp_kses_post($a['html']) : '';
        $excerpt = isset($a['excerpt']) ? wp_strip_all_tags($a['excerpt']) : '';
        $tags    = isset($a['tags']) ? (array)$a['tags'] : [];
        $cat     = isset($a['category']) ? sanitize_text_field($a['category']) : '';
        $img     = isset($a['featured_image_url']) ? esc_url_raw($a['featured_image_url']) : '';
        $incoming_status = isset($a['status']) ? strtolower($a['status']) : '';
        $publish_at = isset($a['publish_at_iso8601']) ? $a['publish_at_iso8601'] : null;
        $canonical = isset($a['canonical_url']) ? esc_url_raw($a['canonical_url']) : '';

        // Default to publish (cron job on Marvlus.ai side sends all articles as Published!)
        $status = 'publish';
        if ($o['trust_incoming_status'] === '1') {
            if (in_array($incoming_status, ['draft','publish','future'], true)) {
                $status = $incoming_status;
            } elseif ($incoming_status === 'publish_now') {
                $status = 'publish';
            } elseif ($incoming_status === 'approved') {
                $status = 'publish'; // Changed from draft to publish per requirements
            }
        }

        $postarr = [
            'post_title'   => $title,
            'post_name'    => $slug ?: sanitize_title($title),
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_status'  => $status,
            'post_author'  => (int)$o['default_author'],
            'post_type'    => 'post',
        ];

        // Schedule if publish_at provided
        if (!empty($publish_at)) {
            try {
                $dt = new DateTime($publish_at, new DateTimeZone('UTC'));
                $postarr['post_status']   = 'future';
                $postarr['post_date_gmt'] = $dt->format('Y-m-d H:i:s');
                $postarr['post_date']     = get_date_from_gmt($postarr['post_date_gmt']);
            } catch (Exception $e) {}
        }

        $post_id = wp_insert_post($postarr, true);
        if (is_wp_error($post_id)) {
            return new WP_REST_Response(['error'=>$post_id->get_error_message()], 500);
        }

        // Tags
        if (!empty($tags)) {
            $tags_clean = array_map('sanitize_text_field', $tags);
            wp_set_post_terms($post_id, $tags_clean, 'post_tag', false);
        }

        // Category or fallback
        if ($cat) {
            $term = term_exists($cat, 'category');
            if (!$term) $term = wp_insert_term($cat, 'category');
            if (!is_wp_error($term) && !empty($term['term_id'])) {
                wp_set_post_terms($post_id, [(int)$term['term_id']], 'category', false);
            }
        } elseif (!empty($o['category_fallback'])) {
            $fb = sanitize_text_field($o['category_fallback']);
            $term = term_exists($fb, 'category');
            if (!$term) $term = wp_insert_term($fb, 'category');
            if (!is_wp_error($term) && !empty($term['term_id'])) {
                wp_set_post_terms($post_id, [(int)$term['term_id']], 'category', false);
            }
        }

        // Featured image
        if ($img) {
            $attach_id = $this->sideload($img, $post_id, $title);
            if ($attach_id && !is_wp_error($attach_id)) {
                set_post_thumbnail($post_id, $attach_id);
            }
        }

        // Optional canonical for later SEO use
        if ($canonical) {
            update_post_meta($post_id, '_contentbeast_canonical_url', $canonical);
        }

        return new WP_REST_Response([
            'ok'        => true,
            'post_id'   => $post_id,
            'status'    => get_post_status($post_id),
            'edit_link' => get_edit_post_link($post_id, ''),
            'published' => $status === 'publish'
        ], 200);
    }

    private function sideload($url, $post_id, $desc = '') {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $tmp = download_url($url);
        if (is_wp_error($tmp)) return $tmp;

        $file_array = [
            'name'     => basename(parse_url($url, PHP_URL_PATH)),
            'tmp_name' => $tmp,
        ];

        $id = media_handle_sideload($file_array, $post_id, $desc);
        if (is_wp_error($id)) {
            @unlink($file_array['tmp_name']);
            return $id;
        }
        return $id;
    }
}

new ContentBeast_Auto_Publisher_Lite();