<?php
/**
 * Plugin Name: ContentBeast Auto Publisher
 * Plugin URI: https://contentbeast.com/wordpress
 * Description: Automated SEO blog writer and publisher that drives endless traffic.
 * Version: 1.3.1
 * Author: ContentBeast
 * Author URI: https://contentbeast.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: contentbeast-auto-publisher
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

/*
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

if (!defined('ABSPATH')) exit;

/**
 * Main plugin class
 * Note: Class name retains '_Lite' suffix for backward compatibility with existing installations
 */

final class ContentBeast_Auto_Publisher_Lite {
    const OPT_GROUP = 'contentbeast_auto_pub_opts_group';
    const OPT_NAME  = 'contentbeast_auto_pub_opts';
    const NS        = 'contentbeast/v1';
    const ROUTE     = '/article';
    const CRON_HOOK = 'contentbeast_daily_publish_cron';

    public function __construct() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'register']);
        add_action('rest_api_init', [$this, 'rest']);
        add_action('init', [$this, 'setup_cron']);
        add_action(self::CRON_HOOK, [$this, 'daily_cron_job']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']);
        
        // Clean up cron on deactivation
        register_deactivation_hook(__FILE__, [$this, 'deactivate_cron']);
    }

    private function opts() {
        $defaults = [
            'contentbeast_api_key'   => '',
            'webhook_url'            => '',
            'default_status'         => 'publish', // Always publish as per requirements
            'default_author'         => 1,       // WP user ID
            'category_fallback'      => '',
            'trust_incoming_status'  => '1',
            'cron_enabled'           => '1',
            'cron_time'              => '02:00',  // 2am CST
            'last_cron_run'          => ''
        ];
        return wp_parse_args(get_option(self::OPT_NAME, []), $defaults);
    }

    public function enqueue_admin_styles($hook) {
        // Only load on our settings page
        if ($hook !== 'settings_page_contentbeast-auto-publisher-lite') {
            return;
        }
        wp_enqueue_style(
            'contentbeast-admin-styles',
            plugins_url('assets/admin.css', __FILE__),
            [],
            '1.3.5'  // Bumped for green theme and text visibility fixes
        );
    }

    public function menu() {
        add_options_page(
            'ContentBeast Publisher',
            'ContentBeast Publisher',
            'manage_options',
            'contentbeast-auto-publisher-lite',
            [$this, 'settings_page']
        );
    }

    public function sanitize_options($input) {
        $sanitized = [];
        
        if (isset($input['contentbeast_api_key'])) {
            $sanitized['contentbeast_api_key'] = sanitize_text_field($input['contentbeast_api_key']);
        }
        
        if (isset($input['webhook_url'])) {
            $sanitized['webhook_url'] = esc_url_raw($input['webhook_url']);
        }
        
        if (isset($input['default_status'])) {
            $sanitized['default_status'] = sanitize_text_field($input['default_status']);
        }
        
        if (isset($input['default_author'])) {
            $sanitized['default_author'] = absint($input['default_author']);
        }
        
        if (isset($input['category_fallback'])) {
            $sanitized['category_fallback'] = sanitize_text_field($input['category_fallback']);
        }
        
        if (isset($input['trust_incoming_status'])) {
            $sanitized['trust_incoming_status'] = sanitize_text_field($input['trust_incoming_status']);
        }
        
        if (isset($input['cron_enabled'])) {
            $sanitized['cron_enabled'] = sanitize_text_field($input['cron_enabled']);
        }
        
        if (isset($input['cron_time'])) {
            $sanitized['cron_time'] = sanitize_text_field($input['cron_time']);
        }
        
        if (isset($input['last_cron_run'])) {
            $sanitized['last_cron_run'] = sanitize_text_field($input['last_cron_run']);
        }
        
        return $sanitized;
    }

    public function register() {
        register_setting(self::OPT_GROUP, self::OPT_NAME, [
            'sanitize_callback' => [$this, 'sanitize_options']
        ]);

        // Main section (no title, just API key and webhook)
        add_settings_section('m_main', '', function () {
            // No description needed
        }, 'contentbeast-auto-publisher-lite');

        // Advanced Settings section (collapsible)
        add_settings_section('m_advanced', '', function () {
            echo '<details style="margin-top: 30px;">';
            echo '<summary style="cursor: pointer; font-weight: 600; font-size: 15px; padding: 10px 0; color: #1d2327;">Advanced Settings</summary>';
            echo '<div style="margin-top: 15px; padding-left: 10px;">';
            echo '<p style="color:#666;font-size:13px;">Optional settings for advanced users.</p>';
        }, 'contentbeast-auto-publisher-lite');

        // API Key field (in main section)
        add_settings_field('contentbeast_api_key', '', function () {
            $o = $this->opts();
            $webhook_url = rest_url(self::NS . self::ROUTE);
            
            echo '<label for="contentbeast_api_key" style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">API Key</label>';
            echo '<input type="password" id="contentbeast_api_key" style="width:420px" name="'.esc_attr(self::OPT_NAME).'[contentbeast_api_key]" value="'.esc_attr($o['contentbeast_api_key']).'" required>';
            echo '<p class="description">Get this from your ContentBeast.com WP integration page</p>';
            
            // Webhook URL right below API key
            echo '<div style="margin-top:20px;">';
            echo '<label style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Your Webhook URL</label>';
            echo '<code style="background:#f1f1f1;padding:10px;display:block;word-break:break-all;">' . esc_html($webhook_url) . '</code>';
            echo '</div>';
            
            // Save button right after
            echo '<div style="margin-top:15px;">';
            echo '<input type="submit" name="submit" class="button button-primary button-large" value="Save Changes">';
            echo '</div>';
        }, 'contentbeast-auto-publisher-lite', 'm_main');

        // Move all other fields to Advanced Settings section
        add_settings_field('default_author', '', function () {
            $o = $this->opts();
            echo '<label for="default_author" style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Default Author (user ID)</label>';
            echo '<input type="number" id="default_author" min="1" style="width:120px" name="'.esc_attr(self::OPT_NAME).'[default_author]" value="'.esc_attr($o['default_author']).'">';
            echo '<p class="description">WordPress user ID for published articles</p>';
        }, 'contentbeast-auto-publisher-lite', 'm_advanced');

        add_settings_field('category_fallback', '', function () {
            $o = $this->opts();
            echo '<label for="category_fallback" style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Fallback Category (name)</label>';
            echo '<input type="text" id="category_fallback" style="width:280px" name="'.esc_attr(self::OPT_NAME).'[category_fallback]" value="'.esc_attr($o['category_fallback']).'">';
            echo '<p class="description">Used when no category is provided in the article data</p>';
        }, 'contentbeast-auto-publisher-lite', 'm_advanced');

        add_settings_field('trust_incoming_status', '', function () {
            $o = $this->opts();
            echo '<label style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Trust incoming status/publish_at</label>';
            ?>
            <label>
                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NAME); ?>[trust_incoming_status]" value="1" <?php checked($o['trust_incoming_status'],'1'); ?>>
                Allow payload fields <code>status</code> and <code>publish_at_iso8601</code> to control draft/publish/schedule
            </label>
        <?php }, 'contentbeast-auto-publisher-lite', 'm_advanced');

        add_settings_field('cron_enabled', '', function () {
            $o = $this->opts();
            echo '<label style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Enable Daily Cron Job</label>';
            ?>
            <label>
                <input type="checkbox" name="<?php echo esc_attr(self::OPT_NAME); ?>[cron_enabled]" value="1" <?php checked($o['cron_enabled'],'1'); ?>>
                Run daily cron job at 2am CST to publish scheduled articles
            </label>
        <?php }, 'contentbeast-auto-publisher-lite', 'm_advanced');

        add_settings_field('technical_reference', '', function () {
            echo '<label style="display:block;font-weight:600;margin-bottom:8px;font-size:14px;">Technical Reference</label>';
            echo '<p style="color:#666;font-size:12px;">Send JSON via POST and include header <code>X-Content-Beast-Signature: sha256=&lt;hmac-hex&gt;</code> where HMAC is computed on the raw request body using your API Key.</p>';
            // Close the collapsible Advanced Settings
            echo '</div>'; // Close padding div
            echo '</details>'; // Close details
        }, 'contentbeast-auto-publisher-lite', 'm_advanced');
    }

    public function settings_page() {
        $o = $this->opts(); 
        $webhook_url = rest_url(self::NS . self::ROUTE);
        $logo_url = plugins_url('contentbeast-logo.png', __FILE__);
        ?>
        <div class="wrap">
            <h1>ContentBeast Auto Publisher</h1>
            
            <!-- ContentBeast Header with Logo -->
            <div class="contentbeast-header">
                <div class="contentbeast-header-content">
                    <img src="<?php echo esc_url($logo_url); ?>" alt="ContentBeast Logo" class="contentbeast-logo">
                    <div class="contentbeast-header-text">
                        <h2>Automate blog writing with AI. Save 2-3 hours a day.</h2>
                        <p>Get mentioned by ChatGPT and rank on Google. Automate high-quality, SEO and AI-optimized blog articles to your website.</p>
                        
                        <!-- Setup Instructions moved here -->
                        <div style="margin-top: 15px;">
                            <h3 style="margin: 0 0 10px 0; font-size: 16px; font-weight: 600;">Setup Instructions</h3>
                            <ol style="margin: 0; padding-left: 20px; font-size: 14px; line-height: 1.8;">
                                <li><strong>Get API Key:</strong> Generate an API key from your <a href="https://contentbeast.com" target="_blank" rel="noopener noreferrer" style="color: #2271b1; text-decoration: none;">ContentBeast.com</a></li>
                                <li><strong>Add API Key:</strong> Enter the API key in the field below and save settings</li>
                                <li><strong>Copy Webhook:</strong> Copy the webhook URL below and add it to your ContentBeast.com settings</li>
                            </ol>
                        </div>
                    </div>
                </div>
            </div>
            
            <form method="post" action="options.php">
                <?php settings_fields(self::OPT_GROUP); ?>
                <?php do_settings_sections('contentbeast-auto-publisher-lite'); ?>
                
                <!-- Bottom Save Button -->
                <input type="submit" name="submit" id="submit-bottom" class="button button-primary button-large" value="Save Changes">
            </form>

            <!-- Hidden curl test section (for debugging) -->
            <details style="margin-top:30px;">
                <summary style="cursor:pointer;color:#666;font-size:13px;">Show test curl command (for developers)</summary>
                <div style="margin-top:10px;">
                    <h3>Test with curl</h3>
<pre style="background:#f5f5f5;padding:15px;overflow-x:auto;font-size:12px;">curl -X POST "<?php echo esc_html($webhook_url); ?>" \
  -H "Content-Type: application/json" \
  -H "X-Content-Beast-Signature: sha256=&lt;hmac-hex&gt;" \
  -d '{
    "event":"article.created",
    "article":{
      "title":"AI Content Strategy for SaaS",
      "slug":"ai-content-strategy-saas",
      "html":"&lt;h1&gt;AI Content Strategy&lt;/h1&gt;&lt;p&gt;...&lt;/p&gt;",
      "excerpt":"Quick guide...",
      "meta_description":"Learn how to create an effective AI content strategy for your SaaS business. Discover proven techniques and best practices.",
      "tags":["ai","saas","seo"],
      "category":"Guides",
      "featured_image_url":"https://example.com/img.jpg",
      "status":"publish",
      "publish_at_iso8601": null,
      "canonical_url":"https://yoursite.com/blog/ai-content-strategy-saas"
    }
  }'</pre>
                    <p style="font-size:12px;color:#666;">Compute HMAC with <code>hash_hmac('sha256', $raw_json, $api_key)</code> and send as hex.</p>
                </div>
            </details>
        </div>
    <?php }

    public function setup_cron() {
        $o = $this->opts();
        
        if ($o['cron_enabled'] === '1') {
            if (!wp_next_scheduled(self::CRON_HOOK)) {
                // Schedule for 2am CST daily
                $timestamp = strtotime('tomorrow 2:00 AM');
                wp_schedule_event($timestamp, 'daily', self::CRON_HOOK);
            }
        } else {
            // Remove cron if disabled
            $timestamp = wp_next_scheduled(self::CRON_HOOK);
            if ($timestamp) {
                wp_unschedule_event($timestamp, self::CRON_HOOK);
            }
        }
    }

    public function deactivate_cron() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            wp_unschedule_event($timestamp, self::CRON_HOOK);
        }
    }

    public function daily_cron_job() {
        $o = $this->opts();
        
        // Update last run time
        $opts = $o;
        $opts['last_cron_run'] = current_time('mysql');
        update_option(self::OPT_NAME, $opts);
        
        // Log cron run for debugging (only when WP_DEBUG is enabled)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('ContentBeast Auto Publisher: Daily cron job executed at ' . current_time('mysql'));
        }
        
        // Here you would typically make an API call to ContentBeast to get scheduled articles
        // For now, we'll just log that the cron ran
        // In a future version, you could add API integration to fetch and publish scheduled content
        
        do_action('contentbeast_daily_cron_executed');
    }

    private function get_next_cron_time() {
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ($timestamp) {
            return date('Y-m-d H:i:s', $timestamp);
        }
        return 'Not scheduled';
    }

    public function rest() {
        register_rest_route(self::NS, self::ROUTE, [
            'methods'  => 'POST',
            'callback' => [$this, 'handle'],
            'permission_callback' => '__return_true'
        ]);
    }

    private function verify_sig($raw, $secret, $header) {
        if (!$secret || !$header) return false;
        $parts = explode('=', $header, 2);
        if (count($parts) !== 2 || strtolower($parts[0]) !== 'sha256') return false;
        $expected = hash_hmac('sha256', $raw, $secret);
        return hash_equals($expected, $parts[1]);
    }

    public function handle(WP_REST_Request $req) {
        $o   = $this->opts();
        $raw = $req->get_body();
        $sig = $req->get_header('x-content-beast-signature');

        if (!$this->verify_sig($raw, $o['contentbeast_api_key'], $sig)) {
            return new WP_REST_Response(['error'=>'invalid_signature'], 401);
        }

        $data = json_decode($raw, true);
        if (!is_array($data) || empty($data['article'])) {
            return new WP_REST_Response(['error'=>'invalid_payload'], 400);
        }

        $a = $data['article'];
        $title   = isset($a['title']) ? wp_strip_all_tags($a['title']) : '(untitled)';
        $slug    = isset($a['slug']) ? sanitize_title($a['slug']) : '';
        $content = isset($a['html']) ? wp_kses_post($a['html']) : '';
        $excerpt = isset($a['excerpt']) ? wp_strip_all_tags($a['excerpt']) : '';
        $meta_description = isset($a['meta_description']) ? wp_strip_all_tags($a['meta_description']) : '';
        $tags    = isset($a['tags']) ? (array)$a['tags'] : [];
        $cat     = isset($a['category']) ? sanitize_text_field($a['category']) : '';
        $img     = isset($a['featured_image_url']) ? esc_url_raw($a['featured_image_url']) : '';
        $incoming_status = isset($a['status']) ? strtolower($a['status']) : '';
        $publish_at = isset($a['publish_at_iso8601']) ? $a['publish_at_iso8601'] : null;
        $canonical = isset($a['canonical_url']) ? esc_url_raw($a['canonical_url']) : '';

        // Default to publish (cron job on Marvlus.ai side sends all articles as Published!)
        $status = 'publish';
        if ($o['trust_incoming_status'] === '1') {
            if (in_array($incoming_status, ['draft','publish','future'], true)) {
                $status = $incoming_status;
            } elseif ($incoming_status === 'publish_now') {
                $status = 'publish';
            } elseif ($incoming_status === 'approved') {
                $status = 'publish'; // Changed from draft to publish per requirements
            }
        }

        $postarr = [
            'post_title'   => $title,
            'post_name'    => $slug ?: sanitize_title($title),
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_status'  => $status,
            'post_author'  => (int)$o['default_author'],
            'post_type'    => 'post',
        ];

        // Schedule if publish_at provided
        if (!empty($publish_at)) {
            try {
                $dt = new DateTime($publish_at, new DateTimeZone('UTC'));
                $postarr['post_status']   = 'future';
                $postarr['post_date_gmt'] = $dt->format('Y-m-d H:i:s');
                $postarr['post_date']     = get_date_from_gmt($postarr['post_date_gmt']);
            } catch (Exception $e) {}
        }

        $post_id = wp_insert_post($postarr, true);
        if (is_wp_error($post_id)) {
            return new WP_REST_Response(['error'=>$post_id->get_error_message()], 500);
        }

        // Tags
        if (!empty($tags)) {
            $tags_clean = array_map('sanitize_text_field', $tags);
            wp_set_post_terms($post_id, $tags_clean, 'post_tag', false);
        }

        // Category or fallback
        if ($cat) {
            $term = term_exists($cat, 'category');
            if (!$term) $term = wp_insert_term($cat, 'category');
            if (!is_wp_error($term) && !empty($term['term_id'])) {
                wp_set_post_terms($post_id, [(int)$term['term_id']], 'category', false);
            }
        } elseif (!empty($o['category_fallback'])) {
            $fb = sanitize_text_field($o['category_fallback']);
            $term = term_exists($fb, 'category');
            if (!$term) $term = wp_insert_term($fb, 'category');
            if (!is_wp_error($term) && !empty($term['term_id'])) {
                wp_set_post_terms($post_id, [(int)$term['term_id']], 'category', false);
            }
        }

        // Featured image
        if ($img) {
            $attach_id = $this->sideload($img, $post_id, $title);
            if ($attach_id && !is_wp_error($attach_id)) {
                set_post_thumbnail($post_id, $attach_id);
            }
        }

        // Optional canonical for later SEO use
        if ($canonical) {
            update_post_meta($post_id, '_contentbeast_canonical_url', $canonical);
        }

        // Store meta description for SEO plugins
        if ($meta_description) {
            // Yoast SEO
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
            
            // Rank Math
            update_post_meta($post_id, 'rank_math_description', $meta_description);
            
            // All in One SEO
            update_post_meta($post_id, '_aioseo_description', $meta_description);
            
            // SEOPress
            update_post_meta($post_id, '_seopress_titles_desc', $meta_description);
            
            // Also use excerpt as fallback if no SEO plugin is active
            if (empty($excerpt)) {
                wp_update_post([
                    'ID' => $post_id,
                    'post_excerpt' => $meta_description
                ]);
            }
        }

        return new WP_REST_Response([
            'ok'        => true,
            'post_id'   => $post_id,
            'status'    => get_post_status($post_id),
            'edit_link' => get_edit_post_link($post_id, ''),
            'published' => $status === 'publish'
        ], 200);
    }

    private function sideload($url, $post_id, $desc = '') {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $tmp = download_url($url);
        if (is_wp_error($tmp)) return $tmp;

        $file_array = [
            'name'     => basename(parse_url($url, PHP_URL_PATH)),
            'tmp_name' => $tmp,
        ];

        $id = media_handle_sideload($file_array, $post_id, $desc);
        if (is_wp_error($id)) {
            @unlink($file_array['tmp_name']);
            return $id;
        }
        return $id;
    }
}

new ContentBeast_Auto_Publisher_Lite();